#include "CSFParser.h"

#include <string>
#include <fstream>

// this function handles all three types of line endings
std::istream& safeGetline(std::istream& is, std::string& t)
{
    t.clear();

    // The characters in the stream are read one-by-one using a std::streambuf.
    // That gives better performance compared with
    // reading them one-by-one using the std::istream.
    // Code that uses streambuf this way must be guarded by a sentry object.
    // The sentry object performs various tasks,
    // such as thread synchronization and updating the stream state.

    std::istream::sentry se(is);
    std::streambuf* sb = is.rdbuf();

    for(;;) {
        int c = sb->sbumpc();
        switch (c) {
        case '\r':
            c = sb->sgetc();
            if(c == '\n')
                sb->sbumpc();
            return is;
        case '\n':
        case EOF:
            return is;
        default:
            t += (char)c;
        }
    }
}

CSFParser::CSFParser(const std::string& filename, size_t offset, bool bRemoveWhitespaces) :
    m_offset(offset),
  m_bRemoveWhitespaces(bRemoveWhitespaces)
{
    std::ifstream stream;
    stream.open(filename.c_str());
    if (!stream.is_open()) return;
    ParseStream(stream);
    stream.close();
}

CSFParser::CSFParser(std::istream& stream, size_t offset, bool bRemoveWhitespaces) :
    m_offset(offset),
    m_bRemoveWhitespaces(bRemoveWhitespaces)
{
    ParseStream(stream);
}

void CSFParser::ParseStream(std::istream& stream) {
    m_RawParsedData.clear();
    do {
        std::string str;
        safeGetline(stream,str);
        if (!str.empty()) ParseLine(str);
    } while (!stream.eof() );
}

std::string CSFParser::TrimStrLeft( const std::string& Src, const std::string& c)
{
      size_t p1 = Src.find_first_not_of(c);
      if (p1 == std::string::npos) {return std::string();}
      return Src.substr(p1);
}

std::string CSFParser::TrimStrRight( const std::string& Src, const std::string& c)
{
      size_t p2 = Src.find_last_not_of(c);
      if (p2 == std::string::npos) {return std::string();}
      return Src.substr(0, p2+1);
}

std::string CSFParser::TrimStr( const std::string& Src, const std::string& c)
{
      size_t p2 = Src.find_last_not_of(c);
      if (p2 == std::string::npos) {return std::string();}
      size_t p1 = Src.find_first_not_of(c);
      if (p1 == std::string::npos) p1 = 0;
      return Src.substr(p1, (p2-p1)+1);
}

std::string CSFParser::cleanup(const std::string& str) {
    if (m_bRemoveWhitespaces) {
        return TrimStr(str);
    } else {
        return str;
    }
}



void CSFParser::ParseLine(const std::string& line) {
    std::vector< std::string > tokenizedLine;

    std::string segment;
    for (size_t pos = 0; pos < line.size(); pos++) {
        if (line[pos] == ',') {
            tokenizedLine.push_back(cleanup(segment));
            segment.clear();
            continue;
        }

        if (line[pos] == '\\') {
            if (pos < line.size()-1) {
                if (line[pos+1] == '\\' ||
                    line[pos+1] == ',') {
                    pos++;
                } else {
                    if (line[pos+1] == 'n') {
                        pos++;
                        segment += "\n";
                        continue;
                    } else {
                        // found unrecognized esc sequence, what should I do?
                        // -> skip it
                        pos++;
                        continue;
                    }
                }
            } else {
                continue;
            }
        }
        segment += line[pos];
    }
    tokenizedLine.push_back(cleanup(segment));

    if (!tokenizedLine.empty())
        m_RawParsedData.push_back(tokenizedLine);
}

size_t CSFParser::SetRecordSizeToMax() {
    if (m_RawParsedData.size() <= m_offset) {
        SetRecordSize(0);
        return 0;
    }
    size_t size = m_RawParsedData[m_offset].size();
    for (size_t i = m_offset+1;i<m_RawParsedData.size();i++) {
        size = std::min(size, m_RawParsedData[i].size());
    }
    SetRecordSize(size);
    return size;
}

void CSFParser::SetVariableRecordSize() {
    m_IndexMapping.clear();
    for (size_t i = m_offset;i<m_RawParsedData.size();i++) {
        m_IndexMapping.push_back(i);
    }
}

void CSFParser::SetRecordSize(size_t iSize) {
    m_IndexMapping.clear();
    for (size_t i = m_offset;i<m_RawParsedData.size();i++) {
        if (m_RawParsedData[i].size() == iSize) {
            m_IndexMapping.push_back(i);
        }
    }
}

const std::vector< std::string >& CSFParser::GetRecord(size_t i) const {
    return m_RawParsedData[m_IndexMapping[i]];
}
