#include <algorithm>
#include "ConferenceUsedata.h"
#include "SciWork.h"
#include "PaperTalk.h"

// 2^31 = 2147483648
#define SESSION_FLAG 2147483648

void ConferenceUsedata::SerializeValues(std::string& serializationString) const {
  // TODO: at the next real DB update remove this hack   
  std::vector<index_t> vEncSelectedScheduleEntries;
  for (size_t i=0;i<m_vSelectedScheduleEntries.size();++i) {
    if (!m_vSelectedScheduleEntries[i].bIsTalk)
      vEncSelectedScheduleEntries.push_back(m_vSelectedScheduleEntries[i].index+SESSION_FLAG);
    else
      vEncSelectedScheduleEntries.push_back(m_vSelectedScheduleEntries[i].index);    
  }

  AddVectorSerialization(serializationString, vEncSelectedScheduleEntries);
  AddVectorSerialization(serializationString, m_vSelectedPapers);
}

bool ConferenceUsedata::DeserializeValues(std::istream& stream) {  
  // TODO: at the next real DB update remove this hack 
  std::vector<index_t> vEncSelectedScheduleEntries;
  if (!DeserializeVector(stream, vEncSelectedScheduleEntries)) return false;
  
  m_vSelectedScheduleEntries.clear();
  for (size_t i=0;i<vEncSelectedScheduleEntries.size();++i) {
    if (vEncSelectedScheduleEntries[i] >= SESSION_FLAG)
      m_vSelectedScheduleEntries.push_back(PersEntry(vEncSelectedScheduleEntries[i]-SESSION_FLAG, false));
    else
      m_vSelectedScheduleEntries.push_back(PersEntry(vEncSelectedScheduleEntries[i], true));
  }  
  
  return DeserializeVector(stream, m_vSelectedPapers);
}

ConferenceUsedata* ConferenceUsedata::CreateOrLoadConferenceUsedataForConferenceDB(const ConferenceDB& confDB, const std::string& fullFilename) {
  ConferenceUsedata* ac;
  
  std::ifstream stream;
  stream.open(fullFilename.c_str());
  if (stream.is_open()) {
    ac = new ConferenceUsedata(stream);
    ac->ValidateAndFix(&confDB);
    stream.close();    
    if (ac->DidConstructionFail()) {
      delete ac;
    } else {
      ac->m_strFilename = fullFilename;
      return ac; 
    }
  }
  
  ac = new ConferenceUsedata();
  ac->m_strFilename = fullFilename;
  ac->m_vSelectedScheduleEntries.clear();
  ac->m_vSelectedPapers.clear();  
  ac->SerializeFile(fullFilename);
  
  return ac;
}


bool ConferenceUsedata::AddScheduleEntry(PersEntry entry, const ConferenceDB* confDB) {
  // check if index is valid
  if (confDB &&
      ((entry.bIsTalk && !confDB->talks.GetEntryByIndex(entry.index)) ||
       (!entry.bIsTalk && !confDB->sessions.GetEntryByIndex(entry.index)))) return false;
  
  // add only if not in list already
  if (std::find(m_vSelectedScheduleEntries.begin(), m_vSelectedScheduleEntries.end(), entry) == m_vSelectedScheduleEntries.end() ) {
    m_vSelectedScheduleEntries.push_back(entry);
    Save();
  }
  
  return true;
}


const SciWork* ConferenceUsedata::AddPaperWithPaperID(index_t paperID, const ConferenceDB* confDB){
  if (!confDB) return NULL;
  // add only if not in list already
  if (std::find(m_vSelectedPapers.begin(), m_vSelectedPapers.end(), paperID) == m_vSelectedPapers.end() ) {
    m_vSelectedPapers.push_back(paperID);
    Save();    
  }
  return confDB->works.GetEntryByIndex(paperID);
}

const PaperTalk* ConferenceUsedata::AddScheduleEntryWithPaperID(index_t paperID, const ConferenceDB* confDB) {
  if (!confDB) return NULL;
  const Paper* p = dynamic_cast<const Paper*>(confDB->works.GetEntryByIndex(paperID));  
  if (!p) return NULL;
  
  const PaperTalk* pt = p->FindContainingTalk(confDB);  
  if (pt) AddScheduleEntry(PersEntry(pt->GetIndex(), true));
  return pt;  
}

bool ConferenceUsedata::Save() {
  return SerializeFile(m_strFilename);
}

void ConferenceUsedata::RemovePaper(index_t paper_index) {
  std::vector<index_t>::iterator iter = std::find(m_vSelectedPapers.begin(), m_vSelectedPapers.end(), paper_index);
  if (iter != m_vSelectedPapers.end() ) {
    m_vSelectedPapers.erase(iter);
    Save();
  }
}

void ConferenceUsedata::RemoveScheduleEntry(PersEntry entry) {    
  std::vector<PersEntry>::iterator iter = std::find(m_vSelectedScheduleEntries.begin(), m_vSelectedScheduleEntries.end(), entry);
  if (iter != m_vSelectedScheduleEntries.end() ) {
    m_vSelectedScheduleEntries.erase(iter);
    Save();
  }
}

void ConferenceUsedata::RemoveScheduleEntryWithPaperID(index_t paperID, const ConferenceDB* confDB) {
  if (!confDB) return;
  
  const SciWork* w = confDB->works.GetEntryByIndex(paperID);
  const Paper* p = dynamic_cast<const Paper*>(w);  
  if (!p) return;
  
  for (size_t i = 0;i<confDB->talks.Count();++i) {
    const Talk* t = confDB->talks.GetEntryByPos(i);
    const PaperTalk* pt = dynamic_cast<const PaperTalk*>(t);    
    if (!pt) continue;
    if (pt->GetPaper() == paperID) {
      RemoveScheduleEntry(PersEntry( pt->GetIndex(), true) );
      return;
    }
  }
}


PersonalSchedule ConferenceUsedata::GetPersonalSchedule(const ConferenceDB* confDB) {
  PersonalSchedule schedule;
  
  // reorder talks into (day, talkslist)-pairs
  for (size_t i = 0;i<m_vSelectedScheduleEntries.size();++i) {
    
    const Session* s = NULL;
    int iSortKey = 0;
    
    if (m_vSelectedScheduleEntries[i].bIsTalk) {
      // get Talk
      const Talk* talk = confDB->talks.GetEntryByIndex(m_vSelectedScheduleEntries[i].index);      
      //////// find day in which this talk is given, therefore ...
      // ... find session
      s = talk->FindContainingSession(confDB);
      iSortKey = talk->GetStartTime();
    } else {
      s = confDB->sessions.GetEntryByIndex(m_vSelectedScheduleEntries[i].index);            
      iSortKey = s->GetStartTime();
    }
    if (!s) continue;
    
    // ... find track
    const Track* t = s->FindContainingTrack(confDB);
    if (!t) continue;
    
    // ... find day
    const WorkDay* d = t->FindContainingDay(confDB);
    if (!d) continue;
    
    // check if we've already seen that day
    size_t index = schedule.size();
    for (size_t j = 0;j<schedule.size();++j) {
      if (schedule[j].first == d->GetIndex()) {
        index = j;
        break;
      }
    }
    
    if (index == schedule.size()) {
      std::vector<PersEntry> indices;
      indices.push_back(m_vSelectedScheduleEntries[i]);
      
      size_t iPos = schedule.size();
      int iSortKey = d->GetDay() + (d->GetMonth()-1) * 40 + d->GetYear() * 500;
      for (size_t j = 0;j<schedule.size();++j) {
        const WorkDay* listDay = confDB->days.GetEntryByIndex(schedule[j].first);
        if (!listDay) continue;
        int iListSortKey = listDay->GetDay() + (listDay->GetMonth()-1) * 40 + listDay->GetYear() * 500;
        
        if (iListSortKey > iSortKey) {
          iPos = j;
          break;
        }
      }
      
      if (iPos == schedule.size()) {
        schedule.push_back(std::make_pair(d->GetIndex(), indices));
      } else {
        schedule.insert(schedule.begin()+iPos, std::make_pair(d->GetIndex(), indices));
      }
    } else {
      size_t iPos = schedule.size();
      for (size_t j = 0;j<schedule[index].second.size();++j) {
        int iListSortKey;
        if (schedule[index].second[j].bIsTalk) {        
          const Talk* listTalk = confDB->talks.GetEntryByIndex(schedule[index].second[j].index);
          if (!listTalk) continue;
          iListSortKey = listTalk->GetStartTime();
        } else {
          s = confDB->sessions.GetEntryByIndex(schedule[index].second[j].index);            
          iListSortKey = s->GetStartTime();          
        }
        
        if (iListSortKey > iSortKey) {
          iPos = j;
          break;
        }
      }
      if (iPos == schedule.size()) {
        schedule[index].second.push_back(m_vSelectedScheduleEntries[i]);
      } else {
        schedule[index].second.insert(schedule[index].second.begin()+iPos, m_vSelectedScheduleEntries[i]);
      }
    }
    
  }
  
  return schedule;
}

void ConferenceUsedata::ValidateAndFix(const ConferenceDB* confDB) {
  for (size_t i = 0;i<m_vSelectedScheduleEntries.size();++i) {
    if (m_vSelectedScheduleEntries[i].bIsTalk) {    
      if (!confDB->talks.GetEntryByIndex(m_vSelectedScheduleEntries[i].index)) 
        m_vSelectedScheduleEntries.erase(m_vSelectedScheduleEntries.begin()+i);
    } else {
      if (!confDB->sessions.GetEntryByIndex(m_vSelectedScheduleEntries[i].index)) 
        m_vSelectedScheduleEntries.erase(m_vSelectedScheduleEntries.begin()+i);      
    }      
  }
  
  for (size_t i = 0;i<m_vSelectedPapers.size();++i) {
    if (!confDB->works.GetEntryByIndex(m_vSelectedPapers[i])) 
      m_vSelectedPapers.erase(m_vSelectedPapers.begin()+i);    
  }
}
