#include <algorithm>

#include "Track.h"
#include "ConferenceDB.h"

Track::Track(const std::string& strTitle) :
Indexable(),
m_strTitle(strTitle)
{
}

Track::Track(const std::string& strTitle, 
                 const std::vector<index_t>& vSessions) :
Indexable(),
m_strTitle(strTitle),
m_vSessions(vSessions)
{
}

Track::Track(std::istream& stream) :
Indexable()
{
  m_bDeserializationSuccess = Deserialize(stream);
}

void Track::SerializeValues(std::string& serializationString) const {
  Indexable::SerializeValues(serializationString);
  AddSerialization(serializationString, m_strTitle);
  AddVectorSerialization(serializationString, m_vSessions);
}

bool Track::DeserializeValues(std::istream& stream) {
  if (!Indexable::DeserializeValues(stream)) return false;
  if (!DeserializeValue(stream, m_strTitle)) return false;
  return DeserializeVector(stream, m_vSessions);
}

void Track::DelSession(index_t index) {
    for (size_t i = 0;i<m_vSessions.size();++i) {
        if (m_vSessions[i] == index)
            m_vSessions.erase(m_vSessions.begin()+i);
    }
}

const WorkDay* Track::FindContainingDay(const ConferenceDB* confDB) const {
  for (size_t i = 0;i<confDB->days.Count();++i) {
    const WorkDay* potentialDay = confDB->days.GetEntryByPos(i);    
    const std::vector<index_t>& tracks = potentialDay->GetTracks();
    if (std::find(tracks.begin(), tracks.end(), m_iIndex) != tracks.end() ) {
      return potentialDay;
    }
  }
  return NULL;
}

bool Track::Validate(const ConferenceDB* confDB) const {
  for (size_t i = 0;i<m_vSessions.size();++i) {
    if (! confDB->sessions.GetEntryByIndex(m_vSessions[i]) ) return false;
  }  
  return true;
}

unsigned int Track::GetReferenceCount(const ConferenceDB* confDB) const {
  unsigned int refs = 0;
  for (size_t i = 0;i<confDB->days.Count();++i) {
    const WorkDay* potentialDay = confDB->days.GetEntryByPos(i);    
    const std::vector<index_t>& tracks = potentialDay->GetTracks();
    if (std::find(tracks.begin(), tracks.end(), m_iIndex) != tracks.end() ) {
      refs++;
    }
  }
  return refs;
}

unsigned int Track::Replace(const Track& newTrack, ConferenceDB* confDB) const {
    unsigned int refs = 0;
    for (size_t i = 0;i<confDB->days.Count();++i) {
      WorkDay* potentialDay = confDB->days.GetEntryByPos(i);
      refs += potentialDay->ReplaceTrack(*this, newTrack);
    }
    return refs;
}

int Track::GetStartTime(const ConferenceDB* confDB) const {
  int start = 24*60;
  for (size_t i = 0;i<m_vSessions.size();++i)
    start = std::min(start, confDB->sessions.GetEntryByIndex(m_vSessions[i])->GetStartTime());
  return start;
}

int Track::GetEndTime(const ConferenceDB* confDB) const {
  int end = 0;
  for (size_t i = 0;i<m_vSessions.size();++i)
    end = std::max(end, confDB->sessions.GetEntryByIndex(m_vSessions[i])->GetEndTime());
  return end;
  
}
