#include "mainwindow.h"

#include <algorithm>
#include <QtGui/QMessageBox>
#include <QtGui/QInputDialog>

#include "QImportDialog.h"

void MainWindow::setupDaysTab(){
    m_tabDays = new QWidget();
    m_tabDays->setObjectName(QString::fromUtf8("m_tabDays"));
    m_tabDays->setWindowTitle(QString::fromUtf8("Days"));
    m_globalTabs->addTab(m_tabDays, m_tabDays->windowTitle());

    m_DayGlobalLayout = new QHBoxLayout(m_tabDays);
    m_DayGlobalLayout->setSpacing(6);
    m_DayGlobalLayout->setContentsMargins(0, 0, 0, 0);
    m_DayGlobalLayout->setObjectName(QString::fromUtf8("m_DayGlobalLayout"));
    m_DaySplitter = new QSplitter();
    m_DayGlobalLayout->addWidget(m_DaySplitter);

    m_DayListFrame = new QFrame(m_tabDays);
    m_DayListFrame->setObjectName(QString::fromUtf8("m_DayListFrame"));
    m_DayListFrame->setFrameShape(QFrame::NoFrame);
    m_DayListFrame->setLineWidth(0);
    m_gridLayoutDayList = new QGridLayout(m_DayListFrame);
    m_gridLayoutDayList->setSpacing(6);
    m_gridLayoutDayList->setContentsMargins(0, 0, 0, 0);
    m_gridLayoutDayList->setObjectName(QString::fromUtf8("m_gridLayoutDayList"));

    m_DayImportBtn = new QPushButton(m_tabDays);
    m_DayImportBtn->setText("Import Days from CSV-File");
    m_gridLayoutDayList->addWidget(m_DayImportBtn, 0, 0, 1, 1);

    m_DayList = new QElemList(m_tabDays,false, false);
    m_DayList->setObjectName(QString::fromUtf8("m_DayList"));
    m_gridLayoutDayList->addWidget(m_DayList, 1, 0, 1, 1);

    m_DayEditFrame = new QFrame(m_tabDays);
    m_DayEditFrame->setObjectName(QString::fromUtf8("m_DayEditFrame"));
    m_DayEditFrame->setFrameShape(QFrame::NoFrame);
    m_DayEditFrame->setFrameShadow(QFrame::Raised);
    m_DayEditFrame->setLineWidth(0);
    m_gridLayoutDay = new QGridLayout(m_DayEditFrame);
    m_gridLayoutDay->setSpacing(6);
    m_gridLayoutDay->setContentsMargins(0, 0, 0, 0);
    m_gridLayoutDay->setObjectName(QString::fromUtf8("m_gridLayoutDay"));

    m_DayNameLabel = new QLabel(m_DayEditFrame);
    m_DayNameLabel->setObjectName(QString::fromUtf8("m_DayNameLabel"));
    m_DayNameLabel->setText(QString::fromUtf8("Date"));
    m_gridLayoutDay->addWidget(m_DayNameLabel, 0, 0, 1, 1);
    m_DayDateEdit = new QDateEdit(m_DayEditFrame);
    m_DayDateEdit->setObjectName(QString::fromUtf8("m_DayDateEdit"));
    m_gridLayoutDay->addWidget(m_DayDateEdit, 0, 1, 1, 1);

    m_DayTrackLabel = new QLabel(m_DayEditFrame);
    m_DayTrackLabel->setObjectName(QString::fromUtf8("m_DayTrackLabel"));
    m_DayTrackLabel->setText(QString::fromUtf8("Tracks"));
    m_gridLayoutDay->addWidget(m_DayTrackLabel, 7, 0, 1, 2);

    m_DayTrackList = new QElemList(m_DayEditFrame, false, false);
    m_DayTrackList->setObjectName(QString::fromUtf8("m_DayTrackList"));
    m_gridLayoutDay->addWidget(m_DayTrackList, 8, 0, 1, 2);

    m_DayEditSpacer = new QSpacerItem(20, 40, QSizePolicy::Minimum, QSizePolicy::Expanding);
    m_gridLayoutDay->addItem(m_DayEditSpacer, 9, 0, 1, 1);

    m_DaySplitter->addWidget(m_DayListFrame);
    m_DaySplitter->addWidget(m_DayEditFrame);

    QObject::connect(m_DayDateEdit, SIGNAL(dateChanged ( const QDate & )), this, SLOT(daysDataChanged()));
    QObject::connect(m_DayDateEdit, SIGNAL(dateChanged ( const QDate & )), this, SLOT(conferenceDataChanged()));

    QObject::connect(m_DayList, SIGNAL(newElement()), this, SLOT(daysNewElement()));
    QObject::connect(m_DayList, SIGNAL(newElement()), this, SLOT(conferenceDataChanged()));
    QObject::connect(m_DayList, SIGNAL(delElement(index_t)), this, SLOT(daysDelElement(index_t)));
    QObject::connect(m_DayList, SIGNAL(delElement(index_t)), this, SLOT(conferenceDataChanged()));
    QObject::connect(m_DayList, SIGNAL(activeElement(index_t)), this, SLOT(daysActiveElement(index_t)));

    QObject::connect(m_DayTrackList, SIGNAL(newElement()), this, SLOT(daysTracksNewElement()));
    QObject::connect(m_DayTrackList, SIGNAL(delElement(index_t)), this, SLOT(daysTracksDelElement(index_t)));
    QObject::connect(m_DayTrackList, SIGNAL(delElement(index_t)), this, SLOT(conferenceDataChanged()));

    QObject::connect(m_DayImportBtn, SIGNAL(clicked()), this, SLOT(daysImportCSV()));

    daysActiveElement(INDEX_EMPTY);
}

void MainWindow::daysImportCSV() {
    if (!SaveCheck("importing days from CSV file")) return;

    std::vector<std::string> properties;
    properties.push_back("Day");
    properties.push_back("Month");
    properties.push_back("Year");

    QImportDialog d(properties, this);
    if (QDialog::Accepted != d.exec()) return;

    std::string filename = d.GetFilename();
    if (!filename.empty()) {
        std::vector<int> choices = d.GetChoices();
        CSFParser csvData(filename,d.GetOffset(),d.RemoveWhitespaces());
        csvData.SetRecordSizeToMax();

        for (size_t i = 0;i<csvData.RecordCount();i++) {
            std::vector< std::string > csvRecord = csvData.GetRecord(i);
            WorkDay b(choices[0]>= 0 ? atoi(csvRecord[choices[0]].c_str()) : 1,
                      choices[1]>= 0 ? atoi(csvRecord[choices[1]].c_str()) : 1,
                      choices[2]>= 0 ? atoi(csvRecord[choices[2]].c_str()) : 1);
            m_pConfDB->days.AddUniqueEntry(b);
        }

        conferenceDataChanged();
        UpdateUI();
    }
}


static bool comp (const std::pair<QDate, index_t>& i, const std::pair<QDate, index_t>& j) {
    return (i.first<j.first);
}

void MainWindow::UpdateDaysUI(index_t newIndex) {
    // sort days by date
    std::vector< std::pair<QDate, index_t> > sortList;
    for (size_t i = 0;i<m_pConfDB->days.Count();++i) {
        const WorkDay* d  = m_pConfDB->days.GetEntryByPos(i);
        QDate date(d->GetYear(), d->GetMonth(), d->GetDay());

        sortList.push_back(std::make_pair(date, d->GetIndex()));
    }


    std::stable_sort(sortList.begin(), sortList.end(), comp);

    std::vector< ElemListEntry > dayList;
    for (size_t i = 0;i<m_pConfDB->days.Count();++i) {
        const WorkDay* d  = m_pConfDB->days.GetEntryByIndex(sortList[i].second);

        QDate date(d->GetYear(), d->GetMonth(), d->GetDay());
        dayList.push_back( ElemListEntry(std::string(date.toString("ddd MMMM d yyyy").toUtf8()),
                                         d->GetIndex()));
    }
    m_DayList->ElemUpdate(dayList, newIndex);

    UpdateDayTrackList();
}

void MainWindow::UpdateDayTrackList() {
    UpdateDayTrackList(m_DayList->GetActiveIndex());
}

void MainWindow::UpdateDayTrackList(index_t index) {
    const WorkDay* d = m_pConfDB->days.GetEntryByIndex(index);

    std::vector< ElemListEntry > sessList;
    if (d) {
        for (size_t i = 0;i<d->GetTracks().size();++i) {
            index_t sessIndex = d->GetTracks()[i];
            const Track* t = m_pConfDB->tracks.GetEntryByIndex(sessIndex);

            std::stringstream ss;
            if (!t->GetTitle().empty())
                ss << t->GetTitle().c_str() << ": ";

            if (t->GetSessions().size()) {
                int start = t->GetStartTime(m_pConfDB);
                int end = t->GetEndTime(m_pConfDB);

                QTime startTime(start/60, start%60);
                QTime endTime(end/60, end%60);

                ss << std::string(startTime.toString("hh:mm ap").toUtf8()).c_str() << " - " << std::string(endTime.toString("hh:mm ap").toUtf8()).c_str();
            } else {
                ss << "no sessions in this track";
            }

            sessList.push_back( ElemListEntry(ss.str(),
                                              t->GetIndex()));
        }
    }
    m_DayTrackList->ElemUpdate(sessList, INDEX_EMPTY);
}

// Slots

void MainWindow::daysDataChanged() {
    if (m_bEditBlockSignals) return;

    index_t index = m_DayList->GetActiveIndex();
    WorkDay* t = m_pConfDB->days.GetEntryByIndex(index);

    t->SetDay(m_DayDateEdit->date().day());
    t->SetMonth(m_DayDateEdit->date().month());
    t->SetYear(m_DayDateEdit->date().year());
    UpdateDaysUI(index);
}

void MainWindow::daysNewElement() {
    WorkDay newDay(1,1, QDate::currentDate().year());
    index_t index = m_pConfDB->days.AddEntry(newDay);
    UpdateDaysUI(index);
    UpdateTracksUI();
}

void MainWindow::daysDelElement(index_t index) {
    WorkDay* t = m_pConfDB->days.GetEntryByIndex(index);
    assert(t);

    m_pConfDB->days.RemoveEntryByIndex(index);
    UpdateDaysUI(INDEX_EMPTY);
    UpdateTracksUI();
}

void MainWindow::daysActiveElement(index_t index) {
    const WorkDay* t = m_pConfDB->days.GetEntryByIndex(index);
    bool bo = StartEditBlocker();
    if (t) {
        m_DayDateEdit->setDate(QDate (t->GetYear(), t->GetMonth(), t->GetDay()) );
        UpdateDayTrackList(index);
        m_DayDateEdit->setEnabled(true);
        m_DayTrackList->setEnabled(true);
    } else {
        m_DayDateEdit->setDate( QDate(QDate::currentDate().year(),1,1 ));
        m_DayTrackList->Clear();
        m_DayDateEdit->setEnabled(false);
        m_DayTrackList->setEnabled(false);
    }
    EndEditBlocker(bo);
}

void MainWindow::daysTracksNewElement() {
    index_t dIndex = m_DayList->GetActiveIndex();
    WorkDay* t = m_pConfDB->days.GetEntryByIndex(dIndex);
    assert(t);

    QStringList items;
    for (size_t i = 0;i<m_pConfDB->tracks.Count();++i) {
        const Track* s = m_pConfDB->tracks.GetEntryByPos(i);
        if (!s) continue;

        // only display tracks that are not in use elsewhere
        if (s->GetReferenceCount(m_pConfDB) == 0) {
            std::stringstream ss;
            ss << s->GetTitle().c_str() << " (ID=" << s->GetIndex() << ")";
            items << ss.str().c_str();
        }
    }

    if (items.count() == 0) {
        QMessageBox msgBox;
        msgBox.setIcon(QMessageBox::Warning);
        msgBox.setText("No additional tracks available.");
        msgBox.exec();
        return;
    }

    items.sort();

    bool ok;
    QString item = QInputDialog::getItem(this, tr("Select an additional Track"),
                                         tr("Tracks:"), items, 0, false, &ok);
    if (ok && !item.isEmpty()) {


        for (size_t i = 0;i<m_pConfDB->tracks.Count();++i) {
            const Track* s = m_pConfDB->tracks.GetEntryByPos(i);
            std::stringstream ss;
            ss << s->GetTitle().c_str() << " (ID=" << s->GetIndex() << ")";
            if (item == QString::fromUtf8( ss.str().c_str() ) ) {
                t->AddTrack(*s);
                break;
            }
        }

        UpdateDaysUI(dIndex);
        UpdateDayTrackList();
        UpdateTracksUI();
        conferenceDataChanged();
    }

}

void MainWindow::daysTracksDelElement(index_t index) {
    index_t tIndex = m_DayList->GetActiveIndex();
    WorkDay* t = m_pConfDB->days.GetEntryByIndex(tIndex);

    t->DelTrack(index);

    UpdateDaysUI(tIndex);
    UpdateTracksUI();
}
