#include "mainwindow.h"

#include <QtGui/QPixmap>
#include <QtGui/QMessageBox>
#include <QtCore/QFileInfo>
#include <QtCore/QDir>
#include <QtGui/QPainter>


bool MainWindow::eventFilter( QObject* watched, QEvent* event ) {
    if ( watched != m_MapImageLabel )
        return false;
    if (!m_MapImageLabel->pixmap())
        return false;

    if ( event->type() == QEvent::MouseButtonPress ) {
        float w = m_MapImageLabel->size().width();
        float h = m_MapImageLabel->size().height();

        if (!w || !h) return false;

        const QMouseEvent* const me = static_cast<const QMouseEvent*>( event );

        //might want to check the buttons here
        const QPoint p = me->pos(); //...or ->globalPos();

        m_RoomXEdit->setValue(std::min(1.0f,p.x()/w));
        m_RoomYEdit->setValue(std::min(1.0f,p.y()/h));

        return true;
    }

    return false;
}


void MainWindow::setupRoomsTab(){
    m_tabRooms = new QWidget();
    m_tabRooms->setObjectName(QString::fromUtf8("m_tabRooms"));
    m_tabRooms->setWindowTitle(QString::fromUtf8("Rooms"));
    m_globalTabs->addTab(m_tabRooms, m_tabRooms->windowTitle());

    m_RoomGlobalLayout = new QHBoxLayout(m_tabRooms);
    m_RoomGlobalLayout->setSpacing(6);
    m_RoomGlobalLayout->setContentsMargins(0, 0, 0, 0);
    m_RoomGlobalLayout->setObjectName(QString::fromUtf8("m_RoomGlobalLayout"));
    m_RoomSplitter = new QSplitter();
    m_RoomGlobalLayout->addWidget(m_RoomSplitter);

    m_RoomList = new QElemList(m_tabRooms, true);
    m_RoomList->setObjectName(QString::fromUtf8("m_RoomList"));
    m_RoomSplitter->addWidget(m_RoomList);

    m_RoomEditFrame = new QFrame(m_tabRooms);
    m_RoomEditFrame->setObjectName(QString::fromUtf8("m_RoomEditFrame"));
    m_RoomEditFrame->setFrameShape(QFrame::NoFrame);
    m_RoomEditFrame->setFrameShadow(QFrame::Raised);
    m_RoomEditFrame->setLineWidth(0);
    m_gridLayoutRoom = new QGridLayout(m_RoomEditFrame);
    m_gridLayoutRoom->setSpacing(6);
    m_gridLayoutRoom->setContentsMargins(0, 0, 0, 0);
    m_gridLayoutRoom->setObjectName(QString::fromUtf8("m_gridLayoutRoom"));

    m_RoomNameLabel = new QLabel(m_RoomEditFrame);
    m_RoomNameLabel->setObjectName(QString::fromUtf8("m_RoomNameLabel"));
    m_RoomNameLabel->setText(QString::fromUtf8("Name"));
    m_gridLayoutRoom->addWidget(m_RoomNameLabel, 0, 0, 1, 1);
    m_RoomNameEdit = new QLineEdit(m_RoomEditFrame);
    m_RoomNameEdit->setObjectName(QString::fromUtf8("m_RoomNameEdit"));
    m_gridLayoutRoom->addWidget(m_RoomNameEdit, 0, 1, 1, 3);

    m_RoomBuildingLabel = new QLabel(m_RoomEditFrame);
    m_RoomBuildingLabel->setObjectName(QString::fromUtf8("m_RoomBuildingLabel"));
    m_RoomBuildingLabel->setText(QString::fromUtf8("Building"));
    m_gridLayoutRoom->addWidget(m_RoomBuildingLabel,1, 0, 1, 1);
    m_RoomBuildingComboBox = new QComboBox(m_RoomEditFrame);
    m_RoomBuildingComboBox->setObjectName(QString::fromUtf8("m_RoomBuildingComboBox"));
    m_gridLayoutRoom->addWidget(m_RoomBuildingComboBox, 1, 1, 1, 1);

    m_RoomFloorIndexLabel = new QLabel(m_RoomEditFrame);
    m_RoomFloorIndexLabel->setObjectName(QString::fromUtf8("m_RoomFloorIndexLabel"));
    m_RoomFloorIndexLabel->setText(QString::fromUtf8("Floor"));
    m_RoomFloorIndexLabel->setAlignment(Qt::AlignLeading|Qt::AlignRight|Qt::AlignCenter);
    m_gridLayoutRoom->addWidget(m_RoomFloorIndexLabel,1, 2, 1, 1);
    m_RoomFloorIndexComboBox = new QComboBox(m_RoomEditFrame);
    m_RoomFloorIndexComboBox->setObjectName(QString::fromUtf8("m_RoomFloorIndexComboBox"));
    m_gridLayoutRoom->addWidget(m_RoomFloorIndexComboBox, 1, 3, 1, 1);

    m_RoomXLabel = new QLabel(m_RoomEditFrame);
    m_RoomXLabel->setObjectName(QString::fromUtf8("m_RoomXLabel"));
    m_RoomXLabel->setText(QString::fromUtf8("Map X"));
    m_gridLayoutRoom->addWidget(m_RoomXLabel, 2, 0, 1, 1);
    m_RoomXEdit = new QDoubleSpinBox(m_RoomEditFrame);
    m_RoomXEdit->setObjectName(QString::fromUtf8("m_RoomXEdit"));
    m_RoomXEdit->setRange(0,1);
    m_RoomXEdit->setSingleStep(0.01);
    m_gridLayoutRoom->addWidget(m_RoomXEdit, 2, 1, 1, 1);

    m_RoomYLabel = new QLabel(m_RoomEditFrame);
    m_RoomYLabel->setObjectName(QString::fromUtf8("m_RoomYLabel"));
    m_RoomYLabel->setText(QString::fromUtf8("Map Y"));
    m_RoomYLabel->setAlignment(Qt::AlignLeading|Qt::AlignRight|Qt::AlignCenter);
    m_gridLayoutRoom->addWidget(m_RoomYLabel, 2, 2, 1, 1);
    m_RoomYEdit = new QDoubleSpinBox(m_RoomEditFrame);
    m_RoomYEdit->setObjectName(QString::fromUtf8("m_RoomYEdit"));
    m_RoomYEdit->setRange(0,1);
    m_RoomYEdit->setSingleStep(0.01);
    m_gridLayoutRoom->addWidget(m_RoomYEdit, 2, 3, 1, 1);

    m_MapImageLabel = new QLabel(m_RoomEditFrame);
    m_MapImageLabel->setObjectName(QString::fromUtf8("m_MapImageLabel"));
    m_MapImageLabel->setScaledContents(true);
    m_MapImageLabel->installEventFilter( this );
 //   m_MapImageLabel->setAlignment(Qt::AlignCenter);
    m_gridLayoutRoom->addWidget(m_MapImageLabel, 3, 0, 1, 4);

    m_RoomEditSpacer = new QSpacerItem(20, 40, QSizePolicy::Minimum, QSizePolicy::Expanding);
    m_gridLayoutRoom->addItem(m_RoomEditSpacer, 4, 0, 1, 1);
    m_RoomSplitter->addWidget(m_RoomEditFrame);

    QObject::connect(m_RoomNameEdit, SIGNAL(textEdited(QString)), this, SLOT(roomsDataChanged()));
    QObject::connect(m_RoomNameEdit, SIGNAL(textEdited(QString)), this, SLOT(conferenceDataChanged()));
    QObject::connect(m_RoomXEdit, SIGNAL(valueChanged(double)), this, SLOT(roomsDataChanged()));
    QObject::connect(m_RoomXEdit, SIGNAL(valueChanged(double)), this, SLOT(conferenceDataChanged()));
    QObject::connect(m_RoomYEdit, SIGNAL(valueChanged(double)), this, SLOT(roomsDataChanged()));
    QObject::connect(m_RoomYEdit, SIGNAL(valueChanged(double)), this, SLOT(conferenceDataChanged()));
    QObject::connect(m_RoomBuildingComboBox, SIGNAL(currentIndexChanged(int)), this, SLOT(roomsBuildingCBChanged()));
    QObject::connect(m_RoomBuildingComboBox, SIGNAL(currentIndexChanged(int)), this, SLOT(conferenceDataChanged()));
    QObject::connect(m_RoomFloorIndexComboBox, SIGNAL(currentIndexChanged(int)), this, SLOT(roomsFloorCBChanged()));
    QObject::connect(m_RoomFloorIndexComboBox, SIGNAL(currentIndexChanged(int)), this, SLOT(conferenceDataChanged()));

    QObject::connect(m_RoomList, SIGNAL(newElement()), this, SLOT(roomsNewElement()));
    QObject::connect(m_RoomList, SIGNAL(delElement(index_t)), this, SLOT(roomsDelElement(index_t)));
    QObject::connect(m_RoomList, SIGNAL(delElement(index_t)), this, SLOT(conferenceDataChanged()));
    QObject::connect(m_RoomList, SIGNAL(activeElement(index_t)), this, SLOT(roomsActiveElement(index_t)));

    roomsActiveElement(INDEX_EMPTY);
}

void MainWindow::renderPosition() {
    const Room* r = m_pConfDB->rooms.GetEntryByIndex(m_RoomList->GetActiveIndex());
    if (!r) return;

    index_t bIndex = r->GetBuilding();
    const Building* b = m_pConfDB->buildings.GetEntryByIndex(bIndex);

    int iFloorIndex = r->GetFloorIndex();
    if (iFloorIndex < 0 || size_t(iFloorIndex) >= b->GetFloorplans().size()) return;

    Floorplan pl = b->GetFloorplans()[iFloorIndex];


    int iMaxW = 600;// std::max(400, m_RoomEditFrame->geometry().width() - 10);
    int iMaxH = 600;//std::max(400, m_RoomEditFrame->geometry().height() - 100);

    QPixmap result(m_BuildingImage);

    QPixmap needle = QPixmap(":/resources/Needle.png");

    QPainter painter(&result);

    int w = result.size().width();
    int h = result.size().height();


    needle = needle.scaledToWidth(needle.size().width()*pl.m_fScale);

    painter.drawPixmap(w*m_RoomXEdit->value() - needle.size().width()*0.5,
                       h*m_RoomYEdit->value() - needle.size().height()*0.97,needle);


   // result = result.scaled(iMaxW, iMaxH,Qt::KeepAspectRatio);

    if (h > 0 && w > 0) {
        if (w/h > iMaxW/iMaxH) {
            m_MapImageLabel->setMaximumSize(iMaxW,iMaxW * h/w);
        } else {
            m_MapImageLabel->setMaximumSize(iMaxH * w/h,iMaxH);
        }
    }
    m_MapImageLabel->setPixmap(result);
}

void MainWindow::UpdateFloorImage(index_t index) {
    QPixmap p;
    m_MapImageLabel->setPixmap(p);
    m_MapImageLabel->setMaximumSize(0,0);

    const Room* r = m_pConfDB->rooms.GetEntryByIndex(index);
    if (!r) return;

    index_t bIndex = r->GetBuilding();
    const Building* b = m_pConfDB->buildings.GetEntryByIndex(bIndex);

    int iFloorIndex = r->GetFloorIndex();
    if (iFloorIndex < 0 || size_t(iFloorIndex) >= b->GetFloorplans().size()) return;

    Floorplan pl = b->GetFloorplans()[iFloorIndex];

    QFileInfo dbFile(m_strCurrentFile);
    QFileInfo target(tr("%1/%2").arg(dbFile.dir().canonicalPath()).arg(pl.m_filename.c_str()));

    if (!target.exists()) return;

    if (m_BuildingImagePath != target.filePath()) {
        m_BuildingImagePath = target.path();
        m_BuildingImage = QPixmap(target.filePath());
    }

    renderPosition();
}

void MainWindow::UpdateRoomsUI() {
    bool b = StartEditBlocker();
    UpdateRoomsUI(m_RoomList->GetActiveIndex());
    UpdateFloorImage(m_RoomList->GetActiveIndex());
    EndEditBlocker(b);
}

void MainWindow::UpdateRoomsUI(index_t newIndex) {
    std::vector< ElemListEntry > rList;
    for (size_t i = 0;i<m_pConfDB->rooms.Count();++i) {
        const Room* r = m_pConfDB->rooms.GetEntryByPos(i);
        rList.push_back( ElemListEntry(r->GetDisplayName(m_pConfDB),
                                         r->GetIndex(),
                                         r->GetReferenceCount(m_pConfDB))  );
    }
    m_RoomList->ElemUpdate(rList, newIndex);
}

void MainWindow::RoomsFillBuildingCB(index_t index) {
    const Room* r =  m_pConfDB->rooms.GetEntryByIndex(index);

    m_RoomBuildingComboBox->clear();
    if (!r) return;

    for (size_t i = 0;i<m_pConfDB->buildings.Count();++i) {
        const Building* b = m_pConfDB->buildings.GetEntryByPos(i);
        if (b->GetFloorplans().size() ==0) continue;
        m_RoomBuildingComboBox->addItem(QString::fromUtf8(b->GetName().c_str()), b->GetIndex());
        if (b->GetIndex() == r->GetBuilding())
            m_RoomBuildingComboBox->setCurrentIndex(m_RoomBuildingComboBox->count()-1);
    }
}

void MainWindow::RoomsFillFloorplanCB(index_t index) {
    const Room* r =  m_pConfDB->rooms.GetEntryByIndex(index);

    m_RoomFloorIndexComboBox->clear();

    if (!r) return;
    const Building* b = m_pConfDB->buildings.GetEntryByIndex(r->GetBuilding());
    assert(b);

    for (size_t i = 0;i< b->GetFloorplans().size();++i) {
        const Floorplan p =  b->GetFloorplans()[i];

        m_RoomFloorIndexComboBox->addItem(QString::fromUtf8(p.m_desc.c_str()), (unsigned int)(i));
        if (i == r->GetFloorIndex())
            m_RoomFloorIndexComboBox->setCurrentIndex(m_RoomFloorIndexComboBox->count()-1);
    }
}

void MainWindow::RoomsFillCBoxes() {
    index_t i = m_RoomList->GetActiveIndex();
    if (!i) return;
    RoomsFillCBoxes(i);
}

void MainWindow::RoomsFillCBoxes(index_t index) {
    bool b = StartEditBlocker();

    RoomsFillBuildingCB(index);
    RoomsFillFloorplanCB(index);
    UpdateFloorImage(index);

    EndEditBlocker(b);
}

// Slots

void MainWindow::roomsDataChanged() {
    if (m_bEditBlockSignals) return;

    index_t index = m_RoomList->GetActiveIndex();
    Room* r =  m_pConfDB->rooms.GetEntryByIndex(index);
    if (!r) return;

    r->SetName(std::string(m_RoomNameEdit->text().toUtf8()));
    r->SetMapPosition( m_RoomXEdit->value(), m_RoomYEdit->value());

    UpdateRoomsUI(index);
    renderPosition();
    UpdateSessionLists();
}

void MainWindow::roomsBuildingCBChanged() {
    if (m_bEditBlockSignals) return;
    index_t index = m_RoomList->GetActiveIndex();
    Room* r =  m_pConfDB->rooms.GetEntryByIndex(index);

    int x = m_RoomBuildingComboBox->currentIndex();
    index_t bIndex = m_RoomBuildingComboBox->itemData(x).toUInt();

    const Building* b = m_pConfDB->buildings.GetEntryByIndex(bIndex);

    assert(b);
    r->SetBuilding(*b);

    RoomsFillFloorplanCB(index);
    UpdateBuildingsUI();
    UpdateBuildingsFloorplans();
    UpdateFloorImage(index);
    UpdateRoomsUI(index);
}

void MainWindow::roomsFloorCBChanged() {
    if (m_bEditBlockSignals) return;
    index_t index = m_RoomList->GetActiveIndex();
    Room* r =  m_pConfDB->rooms.GetEntryByIndex(index);

    int fIndex = m_RoomFloorIndexComboBox->itemData(m_RoomFloorIndexComboBox->currentIndex()).toInt();
    r->SetFloorIndex(fIndex);

    UpdateBuildingsUI();
    UpdateBuildingsFloorplans();
    UpdateFloorImage(index);
    UpdateRoomsUI(index);
}

void MainWindow::roomsNewElement() {
    // find first suitable bulding and floor

    const Building* building = NULL;
    for (size_t i = 0;i<m_pConfDB->buildings.Count();++i) {
        const Building* b = m_pConfDB->buildings.GetEntryByPos(i);
        if (b->GetFloorplans().size()) {
            building = b;
            break;
        }
    }

    if (!building) {
        QMessageBox msgBox;
        msgBox.setIcon(QMessageBox::Critical);
        msgBox.setText("Please create a building with at least one floor first.");
        msgBox.exec();
        return;
    }

    Room newRoom("New", 0, *building, 0, 0);
    index_t index = m_pConfDB->rooms.AddEntry(newRoom);
    UpdateRoomsUI(index);
    UpdateBuildingsUI();
    UpdateBuildingsFloorplans();
    conferenceDataChanged();
}

void MainWindow::roomsDelElement(index_t index) {
    const Room* r =  m_pConfDB->rooms.GetEntryByIndex(index);
    assert(r);
    assert(r->GetReferenceCount(m_pConfDB) == 0);

    m_pConfDB->rooms.RemoveEntryByIndex(index);
    UpdateRoomsUI(index);
    UpdateBuildingsUI();
    UpdateBuildingsFloorplans();
}

void MainWindow::roomsActiveElement(index_t index) {
    const Room* r = m_pConfDB->rooms.GetEntryByIndex(index);
    bool b = StartEditBlocker();
    if (r) {
        m_RoomNameEdit->setText(QString::fromUtf8(r->GetName().c_str()));
        float fMapX, fMapY;
        r->GetMapPosition(fMapX, fMapY);
        m_RoomXEdit->setValue(fMapX);
        m_RoomYEdit->setValue(fMapY);

        RoomsFillCBoxes(index);

        m_RoomNameEdit->setEnabled(true);
        m_RoomXEdit->setEnabled(true);
        m_RoomYEdit->setEnabled(true);
        m_RoomBuildingComboBox->setEnabled(true);
        m_RoomFloorIndexComboBox->setEnabled(true);
    } else {
        m_RoomNameEdit->setText("");
        m_RoomXEdit->setValue(0);
        m_RoomYEdit->setValue(0);
        m_RoomBuildingComboBox->clear();
        m_RoomFloorIndexComboBox->clear();

        RoomsFillCBoxes(INDEX_EMPTY);

        m_RoomNameEdit->setEnabled(false);
        m_RoomXEdit->setEnabled(false);
        m_RoomYEdit->setEnabled(false);
        m_RoomBuildingComboBox->setEnabled(false);
        m_RoomFloorIndexComboBox->setEnabled(false);
    }
    EndEditBlocker(b);
}
